package org.codefilarete.stalactite.sql;

import org.codefilarete.stalactite.engine.SeparateTransactionExecutor;

/**
 * An abstraction for current JDBC connection configuration 
 * 
 * @author Guillaume Mary
 */
public interface ConnectionConfiguration {
	
	ConnectionProvider getConnectionProvider();
	
	/**
	 * Gives the JDBC batch size to use for write operations. Because Stalactite always uses batch, this value is mandatory.
	 * @return the JDBC batch size to use for write operations
	 */
	int getBatchSize();
	
	/**
	 * Gives the JDBC fetch size to use for read operations. Optional, because JDBC drivers have their own. 
	 * @return the JDBC fetch size to use for read operations
	 */
    Integer getFetchSize();
	
    /**
	 * Default implementation of {@link ConnectionConfiguration} that keeps and gives values provided at instantiation time
	 * 
	 * @author Guillaume Mary
	 */
	class ConnectionConfigurationSupport implements ConnectionConfiguration {
		
		private final ConnectionProvider connectionProvider;
		private final int batchSize;
		private final Integer fetchSize;
		
		public ConnectionConfigurationSupport(ConnectionProvider connectionProvider, int batchSize) {
			this(connectionProvider, batchSize, null);
		}

		public ConnectionConfigurationSupport(ConnectionProvider connectionProvider, int batchSize, Integer fetchSize) {
			this.connectionProvider = connectionProvider;
			this.batchSize = batchSize;
			this.fetchSize = fetchSize;
		}
		
		@Override
		public ConnectionProvider getConnectionProvider() {
			return connectionProvider;
		}
		
		@Override
		public int getBatchSize() {
			return batchSize;
		}

		@Override
		public Integer getFetchSize() {
			return fetchSize;
		}
	}
	
	/**
	 * {@link ConnectionProvider} contract that fulfills {@link org.codefilarete.stalactite.engine.PersistenceContext} features 
	 * - entity versioning (pessimistic lock)
	 * - transactional operations (for identifiers generated by {@link org.codefilarete.stalactite.mapping.id.sequence.hilo.PooledHiLoSequence})
	 * 
	 * @author Guillaume Mary
	 */
	interface TransactionalConnectionProvider extends
			TransactionObserver, // for InsertExecutor.VersioningStrategyRollbackListener
			SeparateTransactionExecutor // for PooledHiLoSequence
	{
		
	}
	
}
